#!/usr/bin/python3
from CaseVide import CaseVide
from Sens import Sens
from Direction import Direction
from Mouvement import Mouvement
from File import File
from Arbre import Arbre

class Situation:
    """
    :ivar list liste_voiture: La liste de voitures
    :ivar list grille: La grille représentant une situation
    """

    def __init__(self, liste_voiture):
        """Constructeur de la classe.

        :param list liste_voiture: Une liste de voiture

        >>> from Sens import Sens
        >>> from Taille import Taille
        >>> from Voiture import Voiture
        >>>
        >>> a = Voiture(0,0,"A", Sens.HORIZONTAL, Taille.DEUX)
        >>> b = Voiture(4,4,"B", Sens.HORIZONTAL, Taille.DEUX)
        >>> c = Voiture(2,5,"C", Sens.HORIZONTAL,Taille.TROIS)
        >>> x = Voiture(1,2,"X",Sens.HORIZONTAL,Taille.DEUX)
        >>> situation = Situation([a,b,c,x])
        >>> len(situation.liste_voiture)
        4
        """
        self.liste_voiture = liste_voiture
        self.grille = self.generer_grille(liste_voiture)
        #print(liste_voiture)

    def get_liste_voiture(self):
        """
        Retourne la liste des voitures triées par ordre alphabétique.

        :rtype: ``list(Voiture)``

        >>> from Sens import Sens
        >>> from Taille import Taille
        >>> from Voiture import Voiture
        >>>
        >>> a = Voiture(0,0,"A", Sens.HORIZONTAL, Taille.DEUX)
        >>> b = Voiture(4,4,"B", Sens.HORIZONTAL, Taille.DEUX)
        >>> c = Voiture(2,5,"C", Sens.HORIZONTAL,Taille.TROIS)
        >>> x = Voiture(1,2,"X",Sens.HORIZONTAL,Taille.DEUX)
        >>>
        >>> situation = Situation([a,b,c,x])
        >>> [x.lettre for x in situation.get_liste_voiture()]
        ['A', 'B', 'C', 'X']
        """
        self.liste_voiture.sort(key=lambda x:x.lettre)
        return self.liste_voiture


    def generer_grille(self, liste_voiture):
        """Génère la grille représentant la situation.
        Cette méthode est appelée par le constructeur.
        Elle est composée de :

        * ``Voiture``
        * ``CaseVide``

        :param list liste_voiture: Une liste de voiture
        :rtype: ``list(list())``

        >>> from Sens import Sens
        >>> from Taille import Taille
        >>> from Voiture import Voiture
        >>>
        >>> a = Voiture(0,0,"A", Sens.HORIZONTAL, Taille.DEUX)
        >>> b = Voiture(4,4,"B", Sens.HORIZONTAL, Taille.DEUX)
        >>> c = Voiture(2,5,"C", Sens.HORIZONTAL,Taille.TROIS)
        >>> x = Voiture(1,2,"X",Sens.HORIZONTAL,Taille.DEUX)
        >>>
        >>> situation = Situation([a,b,c,x])
        >>> situation.grille is not None
        True
        """
        grille = list()
        for x in range(0,6):
            l2 = list()
            for y in range(0,6):
                l2.append(CaseVide(x,y))
            grille.append(l2)

        for voiture in liste_voiture:
            for case in voiture.case:
                grille[case[0]][case[1]] = voiture
        return grille

    def afficher(self):
        """Affiche la grille d'une situation

        >>> from Sens import Sens
        >>> from Taille import Taille
        >>> from Voiture import Voiture
        >>>
        >>> a = Voiture(0,0,"A", Sens.HORIZONTAL, Taille.DEUX)
        >>> b = Voiture(4,4,"B", Sens.HORIZONTAL, Taille.DEUX)
        >>> c = Voiture(2,5,"C", Sens.HORIZONTAL,Taille.TROIS)
        >>> x = Voiture(1,2,"X",Sens.HORIZONTAL,Taille.DEUX)
        >>>
        >>> situation = Situation([a,b,c,x])
        >>> situation.afficher()
        [A][A][ ][ ][ ][ ]
        [ ][ ][ ][ ][ ][ ]
        [ ][X][X][ ][ ][ ]
        [ ][ ][ ][ ][ ][ ]
        [ ][ ][ ][ ][B][B]
        [ ][ ][C][C][C][ ]

        """
        s =""
        for y in range(0,6):
            for x in range(0,6):
                print(self.grille[x][y].__str__(),end="")
            print()

    def get_case_vide(self):
        """Retourne la liste des cases vides d'une situation

        :rtype: ``list(CaseVide)``

        >>> from Sens import Sens
        >>> from Taille import Taille
        >>> from Voiture import Voiture
        >>>
        >>> a = Voiture(0,0,"A", Sens.HORIZONTAL, Taille.DEUX)
        >>> b = Voiture(4,4,"B", Sens.HORIZONTAL, Taille.DEUX)
        >>> c = Voiture(2,5,"C", Sens.HORIZONTAL,Taille.TROIS)
        >>> x = Voiture(1,2,"X",Sens.HORIZONTAL,Taille.DEUX)
        >>>
        >>> situation = Situation([a,b,c,x])
        >>> [(c.x,c.y) for c in situation.get_case_vide()]
        [(0, 1), (0, 2), (0, 3), (0, 4), (0, 5), (1, 1), (1, 3), (1, 4), (1, 5), (2, 0), (2, 1), (2, 3), (2, 4), (3, 0), (3, 1), (3, 2), (3, 3), (3, 4), (4, 0), (4, 1), (4, 2), (4, 3), (5, 0), (5, 1), (5, 2), (5, 3), (5, 5)]
        """
        l = list()
        for x in self.grille:
            for y in x:
                if type(y) is CaseVide:
                    l.append(y)
        return l

    def get_mouvements_possibles(self):
        """
        Retourne la liste des mouvements possibles d'une situation.
        Un mouvement est possible quand :

        * une case vide est voisine avec une voiture
        * la case vide est dans un sens adéquat permettant le déplacement d'une voiure

        :rtype: ``list(Mouvement)``

        >>> from Sens import Sens
        >>> from Taille import Taille
        >>> from Voiture import Voiture
        >>>
        >>> a = Voiture(0,0,"A", Sens.HORIZONTAL, Taille.DEUX)
        >>> b = Voiture(4,4,"B", Sens.HORIZONTAL, Taille.DEUX)
        >>> c = Voiture(2,5,"C", Sens.HORIZONTAL,Taille.TROIS)
        >>> x = Voiture(1,2,"X",Sens.HORIZONTAL,Taille.DEUX)
        >>>
        >>> situation = Situation([a,b,c,x])
        >>> situation.afficher()
        [A][A][ ][ ][ ][ ]
        [ ][ ][ ][ ][ ][ ]
        [ ][X][X][ ][ ][ ]
        [ ][ ][ ][ ][ ][ ]
        [ ][ ][ ][ ][B][B]
        [ ][ ][C][C][C][ ]
        >>> [ (m.voiture.lettre,m.direction.name) for m in situation.get_mouvements_possibles()]
        [('X', 'GAUCHE'), ('C', 'GAUCHE'), ('A', 'DROITE'), ('X', 'DROITE'), ('B', 'GAUCHE'), ('C', 'DROITE')]
        """
        from Voiture import Voiture
        mouvement_possible = list() #Dictionnaire voiture:direction possible
        liste_case_vide = self.get_case_vide() #retourne la liste des case vide de la grille
        for case_vide in liste_case_vide:
            liste_voisins= case_vide.get_voisins(self)#recup case voisine d'une case vide// Ca peut etre soit une voiture, soit une case vide
            #Voisin HAUT
            if type(liste_voisins[0]) is Voiture and liste_voisins[0].sens == Sens.VERTICAL:
                mouvement_possible.append(Mouvement(liste_voisins[0], Direction.BAS))
            #Voisin DROITE
            if type(liste_voisins[1]) is Voiture and liste_voisins[1].sens == Sens.HORIZONTAL:
                mouvement_possible.append(Mouvement(liste_voisins[1], Direction.GAUCHE))
            #Voisin BAS
            if type(liste_voisins[2]) is Voiture and liste_voisins[2].sens == Sens.VERTICAL:
                mouvement_possible.append(Mouvement(liste_voisins[2], Direction.HAUT))
            #Voisin GAUCHE
            if type(liste_voisins[3]) is Voiture and liste_voisins[3].sens == Sens.HORIZONTAL:
                mouvement_possible.append(Mouvement(liste_voisins[3], Direction.DROITE))
        return mouvement_possible


    def est_gagnante(self):
        """Indique si la situation est gagnante.

        * ``True`` : La voiture "XX" est aux coordonées [4;2]
        * ``False`` : La voiture "XX" n'est pas en [4:2]

        :rtype: ``boolean``

        >>> from Sens import Sens
        >>> from Taille import Taille
        >>> from Voiture import Voiture
        >>> from Direction import Direction
        >>>
        >>> a = Voiture(0,0,"A", Sens.HORIZONTAL, Taille.DEUX)
        >>> b = Voiture(4,4,"B", Sens.HORIZONTAL, Taille.DEUX)
        >>> c = Voiture(2,5,"C", Sens.HORIZONTAL,Taille.TROIS)
        >>> x = Voiture(1,2,"X",Sens.HORIZONTAL,Taille.DEUX)
        >>>
        >>> situation = Situation([a,b,c,x])
        >>> situation.est_gagnante()
        False
        >>>
        >>> a = Voiture(0,0,"A", Sens.HORIZONTAL, Taille.DEUX)
        >>> b = Voiture(4,4,"B", Sens.HORIZONTAL, Taille.DEUX)
        >>> c = Voiture(2,5,"C", Sens.HORIZONTAL,Taille.TROIS)
        >>> x = Voiture(4,2,"X",Sens.HORIZONTAL,Taille.DEUX)
        >>>
        >>> situation = Situation([a,b,c,x])
        >>> situation.est_gagnante()
        True
        """
        #Voiture XX au coordonée
        if self is None:
            return False

        for t in self.liste_voiture:
            if t.lettre == "X" :
                v = t
        return v.x == 4 and v.y == 2


    def get_situations_suivantes_possibles(self):
        """Retourne la liste des situations suivantes possibles après avoir effectué un seul mouvement.

        Cette liste de tuple est de la forme [ (situation_suivante_possible,mouvement_effectué) ]

        :rtype: ``list(tuple())``

        >>> from Sens import Sens
        >>> from Taille import Taille
        >>> from Voiture import Voiture
        >>> from Direction import Direction
        >>>
        >>> a = Voiture(0,0,"A", Sens.HORIZONTAL, Taille.DEUX)
        >>> b = Voiture(4,4,"B", Sens.HORIZONTAL, Taille.DEUX)
        >>> c = Voiture(2,5,"C", Sens.HORIZONTAL,Taille.TROIS)
        >>> x = Voiture(1,2,"X",Sens.HORIZONTAL,Taille.DEUX)
        >>>
        >>> situation = Situation([a,b,c,x])
        >>> situation.get_situations_suivantes_possibles() # doctest: +SKIP
        """
        situation_suivante_possible = list()
        mouvement_possible = self.get_mouvements_possibles()
        for mouvement in mouvement_possible:
            new_situation = mouvement.execute(self)
            situation_suivante_possible.append( (new_situation, mouvement))
        return situation_suivante_possible

    def __eq__(self,situation):
        """
        Indique l'égalité ou non entre 2 situations.

        On condisère que 2 situations sont identiques si leur liste de voitures sont identiques.

        La méthode retourne donc :

        * ``True`` :  Les deux situations ont la même liste de voitures
        * ``False`` : Les deux situations ont des listes de voitures différentes

        :param Situation situation: Une situation
        :rtype: ``boolean``

        >>> from Sens import Sens
        >>> from Taille import Taille
        >>> from Voiture import Voiture
        >>> from Direction import Direction
        >>>
        >>> a = Voiture(0,0,"A", Sens.HORIZONTAL, Taille.DEUX)
        >>> b = Voiture(4,4,"B", Sens.HORIZONTAL, Taille.DEUX)
        >>> c = Voiture(2,5,"C", Sens.HORIZONTAL,Taille.TROIS)
        >>> x = Voiture(1,2,"X",Sens.HORIZONTAL,Taille.DEUX)
        >>>
        >>> situation_1= Situation([a,b,c,x])
        >>> situation_2= Situation([c,b,a,x])
        >>> situation_3 = Situation([c,b])
        >>> situation_1 == situation_2
        True
        >>> situation_1 == situation_3
        False
        """
        return self.get_liste_voiture() == situation.get_liste_voiture()

    def resoudre(self):
        """Trouve la situation gagnante du Rush Hour.

        Cette méthode génère toutes les situations possibles et construit un arbre au fur et à mesure.

        Chaque noeud de l'arbre est un tuple de la forme ( <situation>,<mouvement_effectue> ).

        La profondeur d'un noeud de l'arbre correspond au nombre de coups nécessaires pour atteindre la situation.

        Un parcours en largeur est effectué pour trouver la situation gagante.

        :rtype: ``Arbre``

        >>> from Sens import Sens
        >>> from Taille import Taille
        >>> from Voiture import Voiture
        >>> from Direction import Direction
        >>>
        >>> a = Voiture(0,0,"A", Sens.HORIZONTAL, Taille.DEUX)
        >>> b = Voiture(4,4,"B", Sens.HORIZONTAL, Taille.DEUX)
        >>> c = Voiture(2,5,"C", Sens.HORIZONTAL,Taille.TROIS)
        >>> x = Voiture(1,2,"X",Sens.HORIZONTAL,Taille.DEUX)
        >>>
        >>> situation= Situation([a,b,c,x])
        >>> arbre = situation.resoudre()
        >>> type(arbre)
        <class 'Arbre.Arbre'>
        """
        deja_fait = []
        prochain = File()
        prochain.enfiler(Arbre((self,None), None))
        stop = False

        while not prochain.est_vide() and not stop:
            current = prochain.defiler() #type(current) = Arbre()
            deja_fait.append(current)
            if current.data[0].est_gagnante():
                #print("FINISH")
                stop=True
            else:
                situation_suivantes = current.data[0].get_situations_suivantes_possibles()
                for situation ,mouvement in situation_suivantes:
                    if situation not in [x.data[0] for x in deja_fait]:
                        if situation not in [x.data[0] for x in prochain.liste]:
                            fils = current.add_fils( (situation,mouvement))
                            prochain.enfiler(fils)
        return current


    def get_mouvements_solution(self, arbre):
        """Retourne la liste ordonée des mouvements à faire pour résoudre le Rush Hour.

        Cette liste de listes est de la forme [ [<mouvement_à_faire>, <occurence>]].

        :param Arbre arbre: Un arbre ayant comme noeux la situation gagnante
        :rtype: ``list(list())``

        >>> from Sens import Sens
        >>> from Taille import Taille
        >>> from Voiture import Voiture
        >>> from Direction import Direction
        >>>
        >>> a = Voiture(0,0,"A", Sens.HORIZONTAL, Taille.DEUX)
        >>> b = Voiture(4,4,"B", Sens.HORIZONTAL, Taille.DEUX)
        >>> c = Voiture(2,5,"C", Sens.HORIZONTAL,Taille.TROIS)
        >>> x = Voiture(1,2,"X",Sens.HORIZONTAL,Taille.DEUX)
        >>>
        >>> situation= Situation([a,b,c,x])
        >>> arbre = situation.resoudre()
        >>> [ (m[0].info(),m[1]) for m in situation.get_mouvements_solution(arbre)]
        [({'voiture': 'X', 'direction': 'Direction.DROITE'}, 3)]

        """

        liste = list()
        while arbre is not None:
            liste.append(arbre.data[1])
            arbre = arbre.parent
        liste.pop()
        liste.reverse()
        liste_compacte = list()
        for i in range(len(liste)):
            if i ==0:
                liste_compacte.append( [liste[i],1])
            else:
                if liste[i].voiture.lettre == liste_compacte[-1][0].voiture.lettre:
                    liste_compacte[-1][1] = liste_compacte[-1][1] +1
                else:
                    liste_compacte.append( [liste[i],1])
        return liste_compacte
