from vehicle import Vehicle


class RushHour:
    VEHICLES_LENGTH = {}
    for v in "ZXABCDEFGHIJK":
        VEHICLES_LENGTH[v] = 2
    for v in "OPQR":
        VEHICLES_LENGTH[v] = 3
    VEHICLES = tuple(VEHICLES_LENGTH.keys())

    def __init__(self):
        """
        create a new (empty) board
        """
        self.__vehicles = dict()
        self.__cells = {(i, j): None for i in range(6)
                        for j in range(6)}

    def state(self) -> bool:
        """
        :return: True if board is valid, False otherwise
        :rtype: bool
        """
        for vname in self.__vehicles:
            v = self.__vehicles[vname]
            for c in v:
                if not self.isCellValid(c, v):
                    return False
        return True

    def isFinal(self) -> bool:
        """
        check if configuration is final
        """
        if "X" in self.__vehicles:
            p = self.__vehicles['X'].pos()
            return p[1] == 4
        else:
            return False

    def fromConf(conf):
        """
        load a board from configuration

        >>> RushHour.fromConf(["X;2,0;H", "O;1,3;V"])
        -------------
        | | | | | | |
        -------------
        | | | |O| | |
        -------------
        |X|X| |O| | |
        -------------
        | | | |O| | |
        -------------
        | | | | | | |
        -------------
        | | | | | | |
        -------------
        """
        board = RushHour()
        for ligne in conf:
            name, pos, orientation = map(lambda s: s.strip(),
                                         ligne.split(";"))
            if name in RushHour.VEHICLES:
                pos = tuple(map(int, pos.split(',')))
                v = Vehicle(name, RushHour.VEHICLES_LENGTH[name],
                            orientation,
                            pos)
                board.append(v)
            else:
                raise Exception(f"vehicle {name} i not valid")
        return board

    def toConf(self) -> set:
        """
        """
        res = set()
        for vname in self.__vehicles:
            v = self.__vehicles[vname]
            res.add("{};{},{};{}".format(v.name(), v.pos()[0],
                                         v.pos()[1],
                                         v.orientation()))
        return res

    def isCellValid(self, c, v=None) -> bool:
        """
        :return: True if c coords are in board and contains v, False otherwise
        :rtype: bool

        Example :
        >>> b = RushHour()
        >>> b.isCellValid((0,0))
        True
        >>> b.isCellValid((6,0))
        False
        """
        return 0 <= c[0] < 6 and 0 <= c[1] < 6 and self.__cells[c] == v

    def append(self, v):
        """
        add vehicle v at position p
        :param v: a vehicle
        :type v: Vehicle
        :param p: a position
        :type p: tuple

        Example :
        >>> r = RushHour()
        >>> r.append(Vehicle("X", 2, "H", (2,0)))
        >>> r
        -------------
        | | | | | | |
        -------------
        | | | | | | |
        -------------
        |X|X| | | | |
        -------------
        | | | | | | |
        -------------
        | | | | | | |
        -------------
        | | | | | | |
        -------------
        """
        for c in v:
            if not self.isCellValid(c):
                raise Exception(f'{v} is out of bounds')
        if v.name() == 'X':
            if v.pos()[0] != 2:
                raise Exception(f'X is not in a suitable place')
        for c in v:
            self.__cells[c] = v
        self.__vehicles[v.name()] = v

    def remove(self, v):
        """
        remove vehicle v from board
        :param v: the vehicle to remove
        :type v: Vehicle

        Example :
        >>> r = RushHour()
        >>> v = Vehicle("X", 2, "H", (2,0))
        >>> r.append(v)
        >>> r.remove(v)
        """
        for c in v:
            self.__cells[c] = None
        del self.__vehicles[v.name()]

    def vehicles(self):
        """
        :return: set of vehicles presents on the board
        :rtype: set of Vehicle

        Example :
        """
        return set(self.__vehicles.keys())

    def vehicle(self, vname):
        return self.__vehicles[vname]

    def cell(self, i, j):
        """
        :return: contents of cell at position (i,j)
        :rtype: Vehicle or None
        """
        return self.__cells[(i, j)]

    def moves(self, v):
        """
        :return: all possible moves for v
        :rtype: str

        Example :
        >>> r = RushHour.fromConf(["X;2,0;H", "O;1,3;V"])
        >>> r.moves(r.cell(2,0))
        'R'
        """
        res = ""
        dirs = Vehicle.ORIENTATION_DIRS[v.orientation()]
        for d in dirs:
            i = 0
            for c in v:
                cp = (c[0]+Vehicle.DIRS[d][0], c[1]+Vehicle.DIRS[d][1])
                if self.isCellValid(cp) or self.isCellValid(cp, v):
                    i += 1
            if i == len(v):
                res += d
        return res

    def move(self, v, d):
        """
        move v in d direction
        """
        if type(v) == str and len(v) == 1:
            v = self.vehicle(v)
        if d not in self.moves(v):
            raise Exception(f"can't move {v} in direction {d}")
        self.remove(v)
        v.move(d)
        self.append(v)

    def playMoves(self, moves):
        """
        play a move sequence
        """
        assert len(moves) % 2 == 0
        i = 0
        while i < len(moves):
            vname = moves[i]
            direction = moves[i+1]
            self.move(self, self.vehicle(vname), direction)
            i += 2

    def allMoves(self):
        """
        get all possible moves
        :return: a dictionnary containing all possible moves
        :rtype: dict
        """
        res = {}
        for vname in self.__vehicles:
            v = self.__vehicles[vname]
            m = self.moves(v)
            if m != '':
                res[vname] = self.moves(v)
        return res

    def __hash__(self) -> int:
        """
        board hash code
        :return: hashcode o the board
        """
        base = len(RushHour.VEHICLES)
        res = 0
        for i in range(6):
            for j in range(6):
                v = self.cell(i, j)
                if v is None:
                    digit = 0
                else:
                    digit = RushHour.VEHICLES.index(v.name())
                res = res * base + digit
        return res

    def __str__(self):
        """
        :return: a string representation of board
        :rtype: str

        Example :
        >>> r = RushHour.fromConf(["X;2,0;H", "O;1,3;V"])
        >>> print(r)
        -------------
        | | | | | | |
        -------------
        | | | |O| | |
        -------------
        |X|X| |O| | |
        -------------
        | | | |O| | |
        -------------
        | | | | | | |
        -------------
        | | | | | | |
        -------------
        """
        line = "-" * 13 + "\n"
        res = line
        for i in range(6):
            for j in range(6):
                v = self.__cells[(i, j)]
                if v is None:
                    res += "| "
                else:
                    res += "|{}".format(v.__str__())
            res += "|\n"
            res += line
        return res[:-1]

    __repr__ = __str__

    def readconfs(confname):
        with open(confname) as f:
            res = []
            conf = []
            line = f.readline()
            while line != '':
                if line == '\n':
                    if conf != "":
                        res.append(conf)
                        conf = []
                else:
                    conf.append(line.strip())
                line = f.readline()
            if conf != []:
                res.append(conf)
        return res


def main():
    import doctest
    doctest.testmod(verbose=True)

    configurations = {'beginner': RushHour.readconfs('beginner.conf'),
                      'expert': RushHour.readconfs('expert.conf')}
    conf = configurations['expert'][-1]
    rh = RushHour.fromConf(conf)
    print(rh)


if __name__ == "  main  ":
    main()
