import pygame
from rushhour import RushHour
from solver import Solver
from time import sleep

SIZE = (600 + 20, 600)
BLACK = (0, 0, 0)
RED = (255, 0, 0)
GREEN = (0, 255, 0)
WHITE = (255, 255, 255)

COLORS = {'X': RED,
          'Z': RED,
          'A': (176, 242, 182),
          'B': (254, 163, 71),
          'C': (15, 157, 232),
          'D': (254, 191, 210),
          'E': (102, 0, 255),
          'F': (0, 142, 142),
          'G': (64, 130, 109),
          'H': (237, 211, 140),
          'I': (255, 255, 0),
          'J': (133, 109, 77),
          'K': (112, 141, 35),
          'O': (255, 203, 96),
          'P': (212, 115, 212),
          'Q': (49, 140, 231),
          'R': (0, 204, 203)}

KEYS = {pygame.key.key_code(letter): letter for letter in COLORS}

KEYS_TO_DIR = {pygame.K_LEFT: 'L',
               pygame.K_RIGHT: 'R',
               pygame.K_UP: 'U',
               pygame.K_DOWN: 'D'}


class PygameRH:
    def __init__(self, game: RushHour):
        pygame.init()
        self._screen = pygame.display.set_mode(SIZE)
        pygame.display.set_caption("Rush Hour")
        self._font = pygame.font.Font('freesansbold.ttf', 20)
        self._letters = {letter: self._font.render(letter, True, BLACK)
                         for letter in COLORS}
        self._game = game
        self._block_width = (SIZE[0]-20) / 6
        self._block_height = SIZE[1] / 6
 
    def draw(self):
        self._screen.fill(BLACK)
        for line in range(6):
            for col in range(6):
                cell = self._game.cell(line, col)
                if cell is not None:
                    x = self._block_width*col
                    y = self._block_height*line
                    pygame.draw.rect(self._screen, COLORS[str(cell)],
                                     pygame.rect.Rect(x, y,
                                                      self._block_width,
                                                      self._block_height))
                    self._screen.blit(self._letters[str(cell)],
                                      (x+self._block_width//2, y+self._block_height//2))
        x = 6*self._block_width
        for line in range(6):
            if line != 2:
                y = self._block_height*line
                pygame.draw.rect(self._screen, GREEN,
                                 pygame.rect.Rect(x, y, 20, self._block_height))
        pygame.display.update()

    def play(self):
        running = True
        v, d = None, None
        old = v, d
        while running:
            for event in pygame.event.get():
                if event.type == pygame.QUIT:
                    running = False
                elif event.type == pygame.KEYDOWN:
                    if event.key in KEYS:
                        v = KEYS[event.key]
                    if event.key in KEYS_TO_DIR:
                        d = KEYS_TO_DIR[event.key]
                    if event.key == pygame.K_s:
                        self._solver = Solver(self._game)
                        self._solver.solve()
                        running = False
                elif event.type == pygame.KEYUP:
                    if event.key in KEYS:
                        v = None
                    if event.key in KEYS_TO_DIR:
                        d = None
            try:
                if v is not None and d is not None and (v, d) != old:
                    self._game.move(v, d)
                old = (v, d)
                running = running and not self._game.isFinal()
            except Exception:
                print(f"can't move {v} in direction {d}")
            self.draw()
        if len(self._solver.solution()) > 0:
            sol = self._solver.solution()[0][-1]
            step = 0
            while not self._game.isFinal():
                v, d = sol[step:step+2]
                self._game.move(v, d)
                self.draw()
                sleep(0.3)
                step += 2
        if self._game.isFinal():
            text = self._font.render('gagné, appuyez sur espace', True, WHITE)
            self._screen.blit(text, ((SIZE[0]-text.get_width())//2, SIZE[1]//2))
            pygame.display.update()
            wait = True
            while wait:
                for event in pygame.event.get():
                    if event.type == pygame.KEYDOWN and \
                       event.key == pygame.K_SPACE:
                        wait = False
        pygame.quit()


def main():
    configurations = {'beginner': RushHour.readconfs('beginner.conf'),
                      'expert': RushHour.readconfs('expert.conf')}
    from random import choice
    conf = choice(configurations['beginner'])
    rh = RushHour.fromConf(conf)
    graphical_rh = PygameRH(rh)
    graphical_rh.play()


if __name__ == '__main__':
    main()
