# -*- coding: utf-8 -*-
'''
:Titre : Jeu de la Vie
:Auteur : L. Conoir
:Date : 12/2019

Simulation de vie cellulaire, animations de motifs (récurrents ou mobiles)

Proposition de correction
'''

from grille import *
from affichage_grille import *
from time import sleep


#####################################################################################################
## Evolution de la grille après une génération
#####################################################################################################
def generation_suivante(grille) :
    '''La fonction prend en paramètre une grille (sous forme de tableau de tableaux) représentant une population de cellules.
Elle renvoie une nouvelle grille représentant le résultat de l'application des règles d'évolution prévues de M.H. Conway.

:param: grille, type list, une grille formé d'un tableau de tableaux
:return: type list, une nouvelle grille formé de tableau de tableaux, après une génération
:CU: grille est de type list
:bord effect: None

:examples:
>>> grille = [[0, 1, 0], [1, 0, 0], [1, 1, 1]]
>>> generation_suivante(grille)
[[0, 0, 0], [1, 0, 1], [1, 1, 0]]
>>> generation_suivante([[0, 0, 0], [1, 0, 1], [1, 1, 0]])
[[0, 0, 0], [1, 0, 0], [1, 1, 0]]
>>> generation_suivante([[0, 0, 0], [1, 0, 0], [1, 1, 0]])
[[0, 0, 0], [1, 1, 0], [1, 1, 0]]
'''

    largeur, hauteur = largeur_grille(grille), hauteur_grille(grille)    # utilisation des fonctions de grille.py
    
    nouvelle_grille = creer_grille(largeur,hauteur)                       # nouvelle grille crrée vide avec une fonction de grille.py
    
    for i in range(hauteur) :
    
        for j in range(largeur) :
            
            nbvoisins = nb_cases_voisins_occup(grille,j,i)                 # détermination du nombre de cellules voisines (grille.py)
            
            if (grille[i][j] == 0 and nbvoisins == 3) :                    # règle de naissance d'une cellule
                nouvelle_grille[i][j] = 1
            
            elif (grille[i][j] == 1 and not(nbvoisins in (2,3))) :         # règles d'extinction d'une cellule
                nouvelle_grille[i][j] = 0
            
            else : nouvelle_grille[i][j] = grille[i][j]                    # dans les autres cas, pas de changement
    
    return nouvelle_grille



#####################################################################################################
## Evolution de la grille après n générations successives
#####################################################################################################
def evolution_n_generations(grille, n) :
    '''La fonction fait évoluer la grille en paramètre après n générations.
Elle renvoie une nouvelle grille représentant le résultat de l'application des règles d'évolution prévues de M.H. Conway.

:param: grille, type list, une grille formée d'un tableau de tableaux
:param: n, type int, nombre de générations
:return: None
:CU: grille est de type list, n est un naturel non nul
:bord effect: la variable grille est modifiée, une fenêtre graphique apparait à l'écran
'''
    dessin = creation_grille(grille)                                   # création de la fenêtre graphique : 'dessin'
    coloration(grille,dessin)                                          # affichage de la grille sur le 'dessin'
    sleep(2)
    
    for i in range(n) :
        
        grille = generation_suivante(grille)                               # grille modifiée pour une génération
        
        coloration(grille,dessin)                                          # affichage de la grille sur le 'dessin'
        
        sleep(0.001)                                                         # temps de pause


#####################################################################################################
## Simulation de l'évolution d'une population de cellules
#####################################################################################################
def simulation(probabilite) :
    '''La fonction affiche la simulation de l'évolution d'une population de cellules qui occupent un espace carré
de 100 cases de côté, avec une probabilité d'occupation fixée en paramètre.
Elle applique les règles d'évolution prévues de M.H. Conway sur 1000 générations.

:param: probabilité, type float, la probabilité d'occupation d'une case par une cellule, initialement.
:return: None
:CU: probabilité est un nombre dans l'intervalle [0;1]
:bord effect: une fenêtre graphique apparait à l'écran
''' 
    grille = creer_grille_aleatoire(70, 70, probabilite)    # cas idéal entre 0.3 et 0.4
    
    evolution_n_generations(grille, 400)
    
        
        
#####################################################################################################
## Motifs récurrents avec nombre de générations et vitesse paramétrés
#####################################################################################################
def motif_recurrent(motif, n, temps) :
    '''La fonction affiche la simulation de l'évolution d'une population de cellules occupant l'espace
défini dans le 'motif'.
Elle applique les règles d'évolution prévues de M.H. Conway sur 'n' générations avec une durée
d'affichage de 'temps' secondes entre chaque génération.

:param: motif, type list, une grille formée d'un tableau de tableaux
:param: n, type int, nombre de générations
:param: temps, type float, temps d'affichage en secondes
:return: None
:CU: motif est de type list, n est un naturel non nul
:bord effect: une fenêtre graphique apparait à l'écran
'''
    dessin = creation_grille(motif)
    for i in range(n) :
        motif = generation_suivante(motif)
        coloration(motif,dessin)
        sleep(temps)


def oscillateur() :
    '''La fonction affiche l'évolution d'un motif "oscillateur".
:param: None
:return: None
:CU: None
:bord effect: une fenêtre graphique apparait à l'écran
'''
    motif = [[0, 0, 1, 0], [1, 0, 0, 1], [1, 0, 0, 1], [0, 1, 0, 0]]
    motif_recurrent(motif,10,0.8)


def galaxie() :
    '''La fonction affiche l'évolution d'un motif "galaxie".
:param: None
:return: None
:CU: None
:bord effect: une fenêtre graphique apparait à l'écran
'''
    motif = [[0] * 15, [0] * 15, [0] * 15,
             [0, 0, 0, 1, 1, 1, 1, 1, 1, 0, 1, 1, 0, 0, 0],
             [0, 0, 0, 1, 1, 1, 1, 1, 1, 0, 1, 1, 0, 0, 0],
             [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0],
             [0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0],
             [0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0],
             [0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0],
             [0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
             [0, 0, 0, 1, 1, 0, 1, 1, 1, 1, 1, 1, 0, 0, 0],
             [0, 0, 0, 1, 1, 0, 1, 1, 1, 1, 1, 1, 0, 0, 0],
             [0] * 15, [0] * 15, [0] * 15]
    motif_recurrent(motif,30,0.8)
    
    
def planeur() :
    '''La fonction affiche l'évolution d'un motif "planeur".
:param: None
:return: None
:CU: None
:bord effect: une fenêtre graphique apparait à l'écran
'''
    motif = [[0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0],
             [0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0],
             [1, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0],
             [0] * 11, [0] * 11, [0] * 11, [0] * 11,
             [0] * 11, [0] * 11, [0] * 11, [0] * 11]
    motif_recurrent(motif,50,0.01)


def canons() :
    '''La fonction affiche l'évolution d'un motif "canons".
:param: None
:return: None
:CU: None
:bord effect: une fenêtre graphique apparait à l'écran
'''
    motif = [[0] * 44,
             [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
             [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
             [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0],
             [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0],
             [0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
             [0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 1, 1, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
             [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
             [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
             [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
             [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
             [0] * 44, [0] * 44, [0] * 44, [0] * 44, [0] * 44, [0] * 44, [0] * 44, [0] * 44, [0] * 44, [0] * 44, [0] * 44, [0] * 44, [0] * 44,
             [0] * 44, [0] * 44, [0] * 44, [0] * 44, [0] * 44, [0] * 44, [0] * 44, [0] * 44, [0] * 44]
    motif_recurrent(motif,400,0.01)



#####################################################################################################
## Insertion du contenu d'une grille dans une autre grille
#####################################################################################################
def insertion(grille1, grille2, x, y) :
    '''La fonction renvoie une grille créée par l'insertion du contenu de 'grille1' dans 'grille2'.
La position d'insertion est définie par le fait que le contenu de la case 0,0 de 'grille1'
est inséré dans la case x,y de grille2.

:param: grille1, grille2, type list, grilles formées d'un tableau de tableaux
:param: x,y, type int, position de départ d'insertion dans 'grille2'
:return: type list, 
:CU: grille1 et grille2 sont de type list, x et y sont des entiers,
     grille1 est de taille inférieure ou égale à grille2 (hauteur et largeur)
:bord effect: Eventuel(s) affichage(s) à l'écran
'''
    from copy import deepcopy
    
    ligne1 = hauteur_grille(grille1)
    colonne1 = largeur_grille(grille1)
    ligne2 = hauteur_grille(grille2)
    colonne2 = largeur_grille(grille2)
    
    grille3 = deepcopy(grille2)                                                # copie de grille2 avec son contenu initiale
    
    troncature = False                                                         # variable précisant si le contenu de grille1 sera tronqué (coupé) lors de l'insertion
    
    if ligne1 <= ligne2 and colonne1 <= colonne2 :                             # condition grille1 plus petite que grille2
        for ligne in range(ligne1) :
            for colonne in range(colonne1) :
                if y + ligne < ligne2 and x + colonne < colonne2 :             # condition : insertion de grille1 ne déborde pas de grille 2
                    grille3[y + ligne][x + colonne] = grille1[ligne][colonne]
                else :
                    troncature = True
    else :
        print("L'insertion est impossible : problème de taille.")
    
    if troncature :
        print("Le contenu de grille1 n'a pas été totalement copié.")
        
    return grille3


#####################################################################################################
## Réalisation de collision à l'aide de motifs prédéfinis
#####################################################################################################
motif_oscillateur = [[0, 0, 1, 0], [1, 0, 0, 1], [1, 0, 0, 1], [0, 1, 0, 0]]
    
motif_galaxie = [[1, 1, 1, 1, 1, 1, 0, 1, 1],
                 [1, 1, 1, 1, 1, 1, 0, 1, 1],
                 [0, 0, 0, 0, 0, 0, 0, 1, 1],
                 [1, 1, 0, 0, 0, 0, 0, 1, 1],
                 [1, 1, 0, 0, 0, 0, 0, 1, 1],
                 [1, 1, 0, 0, 0, 0, 0, 1, 1],
                 [1, 1, 0, 0, 0, 0, 0, 0, 0],
                 [1, 1, 0, 1, 1, 1, 1, 1, 1],
                 [1, 1, 0, 1, 1, 1, 1, 1, 1]]

motif_planeur = [[0, 1, 0],
                 [0, 0, 1],
                 [1, 1, 1]]

motif_canons = [[0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1],
                [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1],
                [1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
                [1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 1, 1, 0, 0, 0, 0, 1, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 0, 0, 0, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 1, 1, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0],
                [0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0]]


def collision(motif1, motif2, colonne, ligne, x1, y1, x2, y2, temps) :
    '''La fonction affiche la collision entre les motifs dans une grille définie par un nombre de colonnes et lignes en paramètre.
La case 0,0 du 'motif1' est positionné dans la case x1,y1 de cette grille.
La case 0,0 du 'motif2' est positionné dans la case x2,y2 de cette grille.

:param: motif1, motif2, type list, grilles formées d'un tableau de tableaux
:param: ligne, colonne, type int, dimension de la grille globale
:param: x1,y1, type int, position de départ de l'insertion du 'motif1'
:param: x2,y2, type int, position de départ de l'insertion du 'motif2'
:param: temps, type float, temps en secondes entre deux mouvements
:return: None
:CU: motif1 et motif2 sont de type list,
     colonne, ligne, x1, y1, x2 et y2 sont des entiers,
     motif1 et motif2 sont de taille inférieure ou égale à la grille définie par colonne et ligne.
:bord effect: une fenêtre graphique apparait à l'écran
'''
    grille = creer_grille(colonne, ligne)         # création de la grille vide
    
    grille = insertion(motif1, grille, x1, y1)     # insertion du motif1 dans la grille
    
    grille = insertion(motif2, grille, x2, y2)     # insertion du motif1 dans la grille
    
    dessin = creation_grille(grille)
    coloration(grille, dessin)
    sleep(2)
    
    for i in range(1000) :
        grille = generation_suivante(grille)
        coloration(grille,dessin)
        sleep(temps)
        


######################################################################
### Vérification des tests mis en examples dans les fonctions
######################################################################
if __name__ == "__main__":
    import doctest, sys
    doctest.testmod()
    simulation(0.1)
    simulation(0.3)
    simulation(0.5)
    simulation(0.7)
    simulation(0.9)
    oscillateur()
    galaxie()
    canons()
    collision(motif_galaxie, motif_planeur, 100, 100, 50, 50, 0, 4, 0.001)
    