'''Module qui permet de dessiner des immeubles
Il utilise le module formes.

Fonctions d'interface du module
-------------------------------

Remarque : il existe quelques couleurs prédéfnies en français qui sont :
    'violette', 'bleue, 'rouge', 'verte', 'jaune'

+ nouvel_immeuble() -> dict : renvoie un dictionnaire contenant les données d'un nouvel immeuble à définir
+ determiner_immeuble(numero:int) -> dict  : renvoie un dictionnaire contenant les caractéristiques aléaloires d'un immeuble 

+ definir_numero_immeuble(numero:int, immeuble:dict) : insére le numéro de l'immeuble dans les données de l'immeuble
+ definir_nb_etages(nb_etages:int, immeuble:dict)    : insère le nombre d'étages de l'immeuble
+ definir_couleur_facade(couleur:str, immeuble:dict) : insère la couleur de la facade
+ definir_position_porte(position:str, immeuble:dict): à choisir parmi 'gauche', 'milieu' ou 'droite'
+ definir_couleur_porte(couleur:str, immeuble:dict)  : insère la couleur de la porte de l'immeuble
+ definir_type_toit(type_toit:str, immeuble:dict)    : à choisir parmi 'plat', 'classique', 'bizarre'

+ dessiner_immeuble(immeuble:dict)  : dessine l'immeuble dont on transmet les données
+ dessiner_rue()                    : dessine une rue de 4 immeubles aléatoires


Remarque : implémentation de l'immeuble
---------------------------------------

L'implémentation choisie par les éléments est souvent un dictionnaire si on réalise l'activité avant la programmation objet.
Ce projet permet alors de montrer que la connaissance des clés ou des indices liés aux attributs de l'immeuble est vitale.
Mais cela manque de centralisation...
Cela donnera une bonne raison lorsqu'on passera à la structure objet d'avoir un endroit où on centralise les attributs.

Pour éviter de passer son temps à courir après les clés, voici la liste des clés d'un immeuble :

--> clé 'numéro' : numéro de l'immeuble (valeur associée de type int)
--> clé 'étages' : nombre d'étages de l'immeuble (valeur associée de type int)
--> clé 'couleur_facade' : couleur de la facade (valeur associée de type str)
--> clé 'position porte' : position de la porte (valeur associée de type int, 0 1 2)
--> clé 'couleur porte' : couleur de la porte (valeur associée de type str)
--> clé 'type toit' : type du toit (valeur associée de type int, 0 pour classique, 1 pour plat, 2 pour bizarre)
--> clé 'couleur toit' : couleur de la porte (valeur associée de type str)


'''


# Importation

import random as rd
from formes import triangle_isocele as triangle
from formes import rectangle


# Constantes

COULEURS_FACADE = {
    'violette': '#AAAACC',
    'bleue': '#88AADD',
    'rouge': '#CCAAAA',
    'verte': '#AACCAA',
    'jaune': '#DDDD66',
    }

COULEURS_PORTE = {
    'violette': '#663366',
    'bleue': '#335577',
    'rouge': '#663333',
    'verte': '#446644',
    'jaune': '#666633',
    }


# Fonctions d'interface 

def nouvel_immeuble():
    '''Renvoie un dictionnaire qui stockera les caractéristiques d'un immeuble "standard" qu'on pourra modifier

    :: return (dict) :: la référence d'un nouveau dictionnaire    
  
    '''
    immeuble = {}
    definir_numero_immeuble(0, immeuble)
    definir_nb_etages(4, immeuble)
    definir_couleur_facade("bleue", immeuble)
    definir_position_porte("milieu", immeuble)
    definir_couleur_porte("violette", immeuble)
    definir_type_toit("classique", immeuble)
    definir_couleur_toit('#333333',immeuble)   
    return immeuble

def definir_numero_immeuble(numero, immeuble):
    '''Modifie immeuble en insérant son numéro dans les informations stockées

    :: param numero(int) :: un entier positif
    :: param immeuble(dict) :: un dictionnaire contenant les données de l'immeuble
    :: return (None)
    .. effet de bord : modifie immeuble
    
    .. exemple ..
    >>> d = nouvel_immeuble()
    >>> definir_numero_immeuble(2, d)
    
    '''
    immeuble['numéro'] = numero

def definir_nb_etages(nb_etages, immeuble):
    '''Modifie immeuble en insérant son nombre d'étages dans les informations stockées

    :: param nb_etages(int) :: un entier positif (1 si plein pied)
    :: param immeuble(dict) :: un dictionnaire contenant les données de l'immeuble
    :: return (None)
    .. effet de bord : modifie immeuble

    .. exemple ..
    >>> d = nouvel_immeuble()
    >>> definir_nb_etages(2, d)
    
    '''
    immeuble['étages'] = nb_etages
  
def definir_couleur_facade(couleur, immeuble):
    '''Modifie immeuble en insérant sa couleur de facade dans les informations stockées

    :: param couleur(str) :: soit une couleur sous forme '#FFFFFF' ou l'une des clés de COULEURS_FACADE
    :: param immeuble(dict) :: un dictionnaire contenant les données de l'immeuble
    :: return (None)
    .. effet de bord : modifie immeuble
    .. remarque : on peut utiliser les clés de COULEURS_FACADE

    .. exemple ..
    >>> d = nouvel_immeuble()
    >>> definir_couleur_facade('#78AACC', d)
    >>> definir_couleur_facade('violette', d)
    
    '''
    if couleur in COULEURS_FACADE.keys():
        immeuble['couleur facade'] = COULEURS_FACADE[couleur]
    else:
        immeuble['couleur facade'] = couleur 
    
def definir_position_porte(position, immeuble):
    '''Modifie immeuble en insérant la position de sa porte

    :: param position(str) :: à choisir parmi 'gauche', 'milieu', 'droite'
    :: param immeuble(dict) :: un dictionnaire contenant les données de l'immeuble
    :: return (None)
    .. effet de bord : modifie immeuble
    
    .. exemple des choix possibles ..
    >>> d = nouvel_immeuble()
    >>> definir_position_porte('gauche', d)
    >>> definir_position_porte('milieu', d)
    >>> definir_position_porte('droite', d)
    
    '''
    if position == 'gauche':
        immeuble['position porte'] = 0
    elif position == 'milieu':
        immeuble['position porte'] = 1
    elif position == 'droite':
        immeuble['position porte'] = 2        
    
def definir_couleur_porte(couleur, immeuble):
    '''Modifie immeuble en insérant sa couleur de porte dans les informations stockées

    :: param couleur(str) :: soit une couleur sous forme '#FFFFFF' ou l'une des clés de COULEURS_PORTE
    :: param immeuble(dict) :: un dictionnaire contenant les données de l'immeuble
    :: return (None)
    .. effet de bord : modifie immeuble
    .. remarque : on peut utiliser les clés de COULEURS_PORTE

    .. exemple ..
    >>> d = nouvel_immeuble()
    >>> definir_couleur_porte('#78AACC', d)
    >>> definir_couleur_porte('violette', d)
    
    '''    
    if couleur in COULEURS_PORTE.keys():
        immeuble['couleur porte'] = COULEURS_PORTE[couleur]
    else:
        immeuble['couleur porte'] = couleur
     
def definir_type_toit(type_de_toit, immeuble):
    '''Modifie immeuble en insérant le type de son toit dans les informations stockées

    :: param type_de_toit(str) :: à choisir parmi 'plat', 'classique' ou 'bizarre'
    :: return (None)
    .. effet de bord : modifie immeuble
    
    .. exemple des choix possibles ..
    >>> d = nouvel_immeuble()
    >>> definir_type_toit('classique', d)
    >>> definir_type_toit('plat', d)
    >>> definir_type_toit('bizarre', d)
    
    '''     
    if type_de_toit == 'classique':
        immeuble['type toit'] = 0
    elif type_de_toit == 'plat':
        immeuble['type toit'] = 1    
    elif type_de_toit == 'bizarre':
        immeuble['type toit'] = 2

def definir_couleur_toit(couleur, immeuble):
    '''Modifie immeuble en insérant le type de son toit dans les informations stockées

    :: param couleur(str) :: une couleur sous forme hexadécimale
    :: return (None)
    .. effet de bord : modifie immeuble
    
    .. exemple ..
    >>> d = nouvel_immeuble()
    >>> definir_couleur_toit('#AA5555', d)
    
    '''     
    immeuble['couleur toit'] = couleur
    

def dessiner_immeuble(immeuble:dict):
    '''Dessine l'immeuble dont on a fourni les caractéristiques

    :: param immeuble(dict) :: un dictionnaire contenant les informations nécessaires
    :: return (None)    
    
    '''
    dessiner_facade(immeuble)
    dessiner_toit(immeuble)
    dessiner_porte(immeuble)
    dessiner_fenetres(immeuble)
    dessiner_toit(immeuble)

def dessiner_rue():
    '''Dessine une rue de 4 immeubles aléatoires

    :: return (None)
    
    '''    
    for x in range(4) :
        info_immeuble = determiner_immeuble(x)  # on génére les infos sur l'immeuble
        dessiner_immeuble(info_immeuble)

def determiner_immeuble(numero):
    '''Renvoie un dictionnaire contenant les attributs aléatoires d'un immeuble dont on connaît le numéro

    :: param numero(int)  :: le numéro de l'immeuble dans la rue
    :: return (dict)      :: la référence d'un nouveau dictionnaire
     
    '''     
    immeuble = nouvel_immeuble()
    immeuble['numéro'] = numero
    
    determiner_facade(immeuble)
    determiner_porte(immeuble)
    determiner_toit(immeuble)
    
    return immeuble



# Fonction gestion des données

def determiner_facade(immeuble):
    '''Modifie l'immeuble en choisissant aléatoirement sa couleur et son nomre d'étages 

    :: param immeuble(dict) :: le dictionnaire contenant les données de l'immeuble
    :: return (None)
    .. effet de bord        .. modifie immeuble
     
    '''
    immeuble['étages'] = choisir_nb_etages()
    immeuble['couleur facade'] = choisir_couleur_facade()
    
def determiner_porte(immeuble):
    '''Modifie l'immeuble en choisissant aléatoirement la position de la porte

    :: param immeuble(dict) :: le dictionnaire contenant les données de l'immeuble
    :: return (None)
    .. effet de bord        .. modifie immeuble
     
    '''     
    immeuble['position porte'] = rd.randint(0,2)
    immeuble['couleur porte'] = choisir_couleur_porte()

def determiner_toit(immeuble):
    '''Modifie l'immeuble en choisissant aléatoirement le type de toit

    :: param immeuble(dict) :: le dictionnaire contenant les données de l'immeuble
    :: return (None)
    .. effet de bord        .. modifie immeuble
     
    ''' 
    immeuble['type toit'] = rd.randint(0,2)
    immeuble['couleur toit'] = '#333333'
    
def choisir_nb_etages():
    '''Modifie l'immeuble en choisissant aléatoirement son nombre d'étages

    :: return (int)   :: un nombre entre 1 (plein pied) et 5 (4 étages)
     
    ''' 
    return rd.randint(1, 5)

def choisir_couleur_facade():
    '''Modifie l'immeuble en choisissant aléatoirement sa couleur de facace

    :: return (str)   :: une couleur sous la forme '#......'
     
    '''   
    couleurs = [couleur for couleur in COULEURS_FACADE.values()]
    choix = rd.randint(0, len(couleurs)-1)
    return couleurs[choix]

def choisir_couleur_porte():
    '''Modifie l'immeuble en choisissant aléatoirement sa couleur de porte

    :: return (str)  :: une couleur sous la forme '#......'
     
    '''   
    couleurs = [couleur for couleur in COULEURS_PORTE.values()]
    choix = rd.randint(0, len(couleurs)-1)
    return couleurs[choix]

def coordonnes_immeuble(immeuble):
    '''Renvoie les coordonnées (x,y) du coin en bas à gauche de l'immeuble

    :: param immeuble(dict)      :: le dictionnaire contenant les données de l'immeuble
    :: return (tuple (int, int)) :: un couple (x,y) des coordonnées en pixels
     
    '''    
    x = immeuble['numéro']*190 - 370
    y = -200
    return (x, y)
    
def decalage_horizontal(position:int) -> int:
    '''Renvoie le décalage (en pixels) à effectuer pour placer l'élément sur la position 0, 1 ou 2 d'un immeuble'''
    return 12.5 + 42.5*position

def decalage_vertical(etage:int) -> int:
    '''Renvoie le décalage (en pixels) à effectuer pour placer l'élément sur l'étage voulu'''
    return 80 * (etage-1)
    


# Fonctions d'interface graphique

def dessiner_facade(immeuble:dict):
    '''Dessiner la facade de l'immeuble transmis'''
    facade = {}  # dictionnaire à transmettre à rectangle
    facade['écriture'] = '#888888'
    facade['fond'] = immeuble['couleur facade']
    facade['épaisseur'] = 3
    
    nb_etages = immeuble['étages']
    x, y = coordonnes_immeuble(immeuble)
    rectangle(140, nb_etages*80, facade, (x,y))
    
def dessiner_porte(immeuble):
    '''Dessiner la porte de l'immeuble transmis'''
    porte = {}  # dictionnaire à transmettre à rectangle
    porte['écriture'] = '#888888'
    porte['fond'] = immeuble['couleur porte']
    porte['épaisseur'] = 3
    
    position = immeuble['position porte']
    xo, yo = coordonnes_immeuble(immeuble)
    decalage = decalage_horizontal(position)
    x = xo + decalage
    y = yo
    rectangle(30, 50, porte, (x,y))
    
def dessiner_toit(immeuble):
    '''Dessiner le toit de l'immeuble transmis'''
    toit = {}  # dictionnaire à transmettre à fonctions de dessin
    toit['écriture'] = '#888888'
    toit['fond'] = '#333333'
    toit['épaisseur'] = 3
    
    xo, yo = coordonnes_immeuble(immeuble)    
    decalage_y = decalage_vertical(immeuble['étages']+1)
    x = xo
    y = yo + decalage_y
    
    type_toit = immeuble['type toit'] 
    if type_toit == 0:
        triangle(140, 35, toit, (x,y))
    elif type_toit == 1:
        rectangle(140+10, 15, toit, (x-5,y))
    elif type_toit == 2:
        triangle(140, 30, toit, (x,y))
        rectangle(140+10, 15, toit, (x-5,y))

def dessiner_fenetres(immeuble):
    '''Dessiner les fenêtres de l'immeuble transmis'''
    fenetres = {}  # dictionnaire à transmettre à dessiner_fenetre
    fenetres['écriture'] = '#888888'
    fenetres['fond'] = '#AAAAFF'
    fenetres['position porte'] = immeuble['position porte']
    fenetres['épaisseur'] = 3
    fenetres['numéro'] = immeuble['numéro']

    xo, yo = coordonnes_immeuble(immeuble) 
    etages = immeuble['étages'] 
    for etage in range(1, etages+1):
        for position in range(3) :
            dessiner_fenetre(fenetres, position, etage, xo, yo)

def dessiner_fenetre(fenetres, position, etage, xo, yo):
    '''Dessiner une fenêtre

    :: param fenetres(dict) :: un dictionnaire regroupant les informations utiles sur les fenêtres(voir dessiner_fenetres() )
    :: param position(int)  :: la position 0, 1 ou 2 de la fenêtre sur l'immeuble
    :: param etage(int)     :: l'étage de la fenêtre (1 pour rez-de-chaussé...)
    :: param xo(int)        :: coordonnées horizontal du coin gauche du bas de l'immeuble
    :: param yo(int)        :: coordonnées vertical du coin gauche du bas de l'immeuble
    
    '''
    if not (etage == 1 and position == fenetres['position porte']):        
        decalage_x = decalage_horizontal(position)
        decalage_y = decalage_vertical(etage)        
        x = xo + decalage_x
        y = yo + decalage_y + 20
        rectangle(30, 30, fenetres, (x,y))
    


# Programme-test

if __name__ == '__main__':
    import doctest
    doctest.testmod()
    dessiner_rue()




    