'''Module qui permet de dessiner des immeubles en écrivant un texte formaté.
Il utile le module rue.

Utilisation
-----------

1 - On crée un fichier texte formaté précisement (voir exemple ci-dessous).
2 - On lance l'interprétation du texte avec la fonction interpreter_dessin()

>>> interpreter_dessin('exemple2.txt')



exemple1.txt : 4 immeubles aléatoires
------------
rue
4 immeubles
aléatoire

exemple2.txt : 4 immeubles mais un seul aléatoire
------------
rue
4 immeubles
facade rouge - 2 étages - porte rouge au milieu - toit classique
aléatoire
facade verte - 4 étages - porte verte à droite - toit plat
facade violet - 3 étages - porte bleue au milieu - toit bizarre

exemple3.txt : 3 immeubles totalement choisis
------------
rue
3 immeubles
facade rouge - 2 étages - porte rouge au milieu - toit classique
facade verte - 4 étages - porte verte à droite - toit plat
facade violet - 3 étages - porte bleue au milieu - toit bizarre

'''

# Importations

from rue import nouvel_immeuble
from rue import definir_numero_immeuble
from rue import definir_nb_etages
from rue import definir_couleur_facade
from rue import definir_position_porte
from rue import definir_couleur_porte
from rue import definir_type_toit
from rue import dessiner_immeuble
from rue import determiner_immeuble

# Fonction d'interface

def interpreter_dessin(nom_fichier):
    '''Lit et interprète le fichier texte dont le nom est transmis'''
    theme, infos = recuperer_informations_dessin("dessin.txt")    
    if theme == 'rue':        
        interpreter_rue(infos)



# Déclaration des fonctions internes

def recuperer_informations_dessin(nom_fichier):
    '''Renvoie les informations qu'on parvient à lire dans le fichier dont le nom est transmis

    :: param nom_fichier(str)  :: le nom du fichier-texte à interpréter
    :: return (tuple(str,list) :: un couple contenant le type du dessin et un tableau contenant les informations connues sur les élémnts du dessin
    
    .. exemple avec l'exemple 1 : recuperer_informations_dessin('exemple1.txt') renvoie
    ('rue', [{'numéro': 0}, {'numéro': 1}, {'numéro': 2}, {'numéro': 3}])

    '''    
    reponse = (None, [])  # réponse par défaut si rien n'est interprétable
    fichier = open(nom_fichier, 'r', encoding="utf-8")
    ligne = fichier.readline().replace('\n','')
    if ligne.split(' ')[0] == 'rue':
        description_rue = lire_rue(fichier)
        reponse = ('rue', description_rue)

    fichier.close()
    
    return reponse

def lire_rue(fichier):
    '''Lit les lignes du fichier et renvoie un tableau de dictionnaire contenant les informations sur les immeubles

    :: param fichier(_io.TextIOWrapper) :: la référence d'un fichier en cours de lecture
    :: return (list)  :: un tableau contenant un dictionnaire par immeuble et contenant les informations lues
    
    .. exemple de réponse lors d'interprétation de l'exemple 3 :
    [
        {'numéro': 0, 'couleur de facade': 'rouge', "nombre d'étages": '2', 'couleur de la porte': 'rouge', 'position de la porte': 'milieu', 'type de toit': 'classique'},
        {'numéro': 1, 'couleur de facade': 'verte', "nombre d'étages": '4', 'couleur de la porte': 'verte', 'position de la porte': 'droite', 'type de toit': 'plat'},
        {'numéro': 2, 'couleur de facade': 'violet', "nombre d'étages": '3', 'couleur de la porte': 'bleue', 'position de la porte': 'milieu', 'type de toit': 'bizarre'}
    ]
    
    '''
    ligne = fichier.readline().replace('\n', '')
    mots = ligne.split(' ')    
   
    rue = []   
    if mots[1] == 'immeuble' or mots[1] == 'immeubles':
        for nb in range(int(mots[0])):
            informations_immeuble = lire_immeuble(fichier, nb)
            rue.append(informations_immeuble)

    return rue

def lire_immeuble(fichier, nb):
    '''Lit les informations de l'immeuble numéro nb dans le fichier et renvoie un dictionnaire contenant les informations récupérées

    :: param fichier(_io.TextIOWrapper) :: la référence d'un fichier en cours de lecture
    :: return (dict)  :: un dictionnaire par immeuble et contenant les informations lues
    
    .. exemple de réponse lors d'interprétation du premier immeuble de l'exemple 3 :
    {'numéro': 0, 'couleur de facade': 'rouge', "nombre d'étages": '2', 'couleur de la porte': 'rouge', 'position de la porte': 'milieu', 'type de toit': 'classique'},
  
    '''
    enregistrement = {}  # Dictionnaire qui va récupérer les informations lues
    enregistrement['numéro'] = nb
    ligne = fichier.readline().replace('\n', '')
    indications = ligne.split('-')
    
    if indications != 'aléatoire':

        for indication in indications:
            if 'facade' in indication:
                enregistrement['couleur de facade'] = lire_information(indication, ['facade'])   
            if 'étage' in indication:
                enregistrement["nombre d'étages"] = lire_information(indication, ['étages', 'étage'])    
            if 'porte' in indication:
                enregistrement["couleur de la porte"] = lire_information(indication, ['porte', 'à', 'au', 'gauche', 'milieu', 'droite'])
                enregistrement["position de la porte"] = lire_information(indication, ['porte', 'à', 'au', enregistrement["couleur de la porte"]])  
            if 'toit' in indication:
                enregistrement["type de toit"] = lire_information(indication, ['toit'])

    return enregistrement
        
def lire_information(indication, mots_a_exclure):
    '''Renvoie un string contenant les informations extraites après avoir exclue les mots qu'on ne veut pas récupérer

    :: param indication(str)       :: une des sous-chaînes de caractères d'une ligne décrivant un immeuble
    :: param mots_a_exclure(list)  :: un tableau contenant les mots qu'on veut exclure de la réponse

    .. exemples
    >>> lire_information(' facade rouge  ', ['facade'])
    rouge
    >>> lire_information('  2 étages  ', ['étages', 'étage'])
    2
    >>> lire_information('  porte rouge au milieu  ', ['porte', 'à', 'au', 'gauche', 'milieu', 'droite'])
    rouge
    >>> lire_information('  porte rouge au milieu  ', ['porte', 'à', 'au', 'rouge'])
    milieu    
    >>> lire_information('  toit classique ', ['toit'])
    classique         
    
    '''
    mots = indication.split(' ')
    mots = [mot for mot in mots if mot !='' and not mot in mots_a_exclure]
    reponse = ''
    nb_mots = len(mots)
    for i in range(nb_mots):
        if i != nb_mots - 1:
            reponse = reponse + mots[i] + ' '
        else:
            reponse = reponse + mots[i]

    return reponse 

def traduire(infos):
    '''Crée et renvoie une structure de données d'immeubles utilisable par le module rue

    :: param infos(list)  :: un tableau contenant les dictionnaires stockant les informations récupérées sur les immmeubles
    :: return (list)      :: un tableau contenant des structures de données d'immeubles du module rue
    
    '''    
    immeubles = []    
    for enregistrement in infos:
        if len(enregistrement) == 1:  # cas qui correspond à un immeuble aléatoire
            immeuble = determiner_immeuble(enregistrement["numéro"])
        else:
            immeuble = nouvel_immeuble()
            definir_numero_immeuble(enregistrement["numéro"], immeuble)
            definir_nb_etages(int(enregistrement["nombre d'étages"]), immeuble)
            definir_couleur_facade(enregistrement["couleur de facade"], immeuble)
            definir_position_porte(enregistrement["position de la porte"], immeuble)
            definir_couleur_porte(enregistrement["couleur de la porte"], immeuble)
            definir_type_toit(enregistrement["type de toit"], immeuble)
        immeubles.append(immeuble)
    
    return immeubles    

def interpreter_rue(infos):
    '''Trace les immeubles à partir d'un tableau de dictionnaires'''
    immeubles = traduire(infos)  # pour rendre les informations compatibles avec le module rue
    for immeuble in immeubles:
        dessiner_immeuble(immeuble)
        
 
if __name__ == '__main__':
    
    interpreter_dessin('dessin.txt')